<?php
class LicenseVerifier {
    private $license_key;
    private $domain;
    private $verification_url;
    private $cache_file;
    private $cache_duration = 3600; // 1 hour

    public function __construct($license_key, $domain, $verification_url) {
        $this->license_key = $license_key;
        $this->domain = $domain;
        $this->verification_url = $verification_url;
        $this->cache_file = sys_get_temp_dir() . '/license_' . md5($license_key) . '.cache';
    }

    public function verify() {
        // Check cache first
        if ($this->checkCache()) {
            return true;
        }

        // Perform verification
        $ch = curl_init($this->verification_url);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, [
            'license_key' => $this->license_key,
            'domain' => $this->domain
        ]);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($http_code !== 200) {
            return false;
        }

        $result = json_decode($response, true);
        
        if ($result['status'] === 'success') {
            // Cache the successful verification
            $this->updateCache();
            return true;
        }

        return false;
    }

    private function checkCache() {
        if (!file_exists($this->cache_file)) {
            return false;
        }

        $cache_data = json_decode(file_get_contents($this->cache_file), true);
        if (!$cache_data) {
            return false;
        }

        // Check if cache is still valid
        if (time() - $cache_data['timestamp'] > $this->cache_duration) {
            unlink($this->cache_file);
            return false;
        }

        return true;
    }

    private function updateCache() {
        $cache_data = [
            'timestamp' => time(),
            'license_key' => $this->license_key,
            'domain' => $this->domain
        ];
        file_put_contents($this->cache_file, json_encode($cache_data));
    }
}

// Example usage:
/*
$verifier = new LicenseVerifier(
    'YOUR-LICENSE-KEY',
    $_SERVER['HTTP_HOST'],
    'http://your-license-server.com/verify.php'
);

if (!$verifier->verify()) {
    die('Invalid license or domain. Please contact support.');
}
*/
?> 