<?php
require_once 'config.php';

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['status' => 'error', 'message' => 'Method not allowed']);
    exit();
}

$license_key = sanitizeInput($_POST['license_key'] ?? '');
$domain = sanitizeInput($_POST['domain'] ?? '');

if (empty($license_key) || empty($domain)) {
    http_response_code(400);
    echo json_encode(['status' => 'error', 'message' => 'License key and domain are required']);
    exit();
}

try {
    // Check if license exists and is valid
    $stmt = $pdo->prepare("SELECT * FROM licenses WHERE license_key = ? AND status = 'active'");
    $stmt->execute([$license_key]);
    $license = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$license) {
        echo json_encode(['status' => 'error', 'message' => 'Invalid or inactive license']);
        exit();
    }

    // Check if license is expired
    if ($license['expires_at'] && strtotime($license['expires_at']) < time()) {
        $stmt = $pdo->prepare("UPDATE licenses SET status = 'expired' WHERE id = ?");
        $stmt->execute([$license['id']]);
        echo json_encode(['status' => 'error', 'message' => 'License has expired']);
        exit();
    }

    // Check if domain matches
    if ($license['domain'] !== $domain) {
        // Log the attempt
        $stmt = $pdo->prepare("INSERT INTO license_logs (license_id, action, ip_address, user_agent) VALUES (?, 'invalid_domain', ?, ?)");
        $stmt->execute([$license['id'], $_SERVER['REMOTE_ADDR'], $_SERVER['HTTP_USER_AGENT']]);
        
        echo json_encode(['status' => 'error', 'message' => 'Invalid domain for this license']);
        exit();
    }

    // Update last check time
    $stmt = $pdo->prepare("UPDATE licenses SET last_check = CURRENT_TIMESTAMP WHERE id = ?");
    $stmt->execute([$license['id']]);

    // Log successful verification
    $stmt = $pdo->prepare("INSERT INTO license_logs (license_id, action, ip_address, user_agent) VALUES (?, 'verified', ?, ?)");
    $stmt->execute([$license['id'], $_SERVER['REMOTE_ADDR'], $_SERVER['HTTP_USER_AGENT']]);

    echo json_encode([
        'status' => 'success',
        'message' => 'License is valid',
        'data' => [
            'expires_at' => $license['expires_at'],
            'domain' => $license['domain']
        ]
    ]);

} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['status' => 'error', 'message' => 'Internal server error']);
}
?> 